/**
 * \file: exchnd_collector.h
 *
 * Collector definition
 * Some data collected in the exception handler is collected in the user
 * space. The file defines the array via which the collection functions
 * are addressed.
 *
 * \component: Exception handler
 *
 * \author: M.Weise
 *
 * \copyright (c) 2010, 2011 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 *
 * \history
 * 2013/12/05 Initial version
 *
 ***********************************************************************/
#ifndef __EXCHND_COLLECTOR_H__
#define __EXCHND_COLLECTOR_H__

#include <libunwind-ptrace.h>

#include <linux/exchnd.h>

#include "exchnd_backend.h"
#include "exchnd_interface.h"

#define BASE_DEC 10
#define BASE_HEX 16

#define MAX_THREADS 1000
#define MAX_RECURSE_COUNT 512

#define TEN_MSECS    10000
#define MAX_RETRIES 10

#if __WORDSIZE == 64
#   define TMPL_REPEAT  "--- Address [0x%lx] repeated %d times ---\n"
#   define TMPL_BT      "%s+0x%lx [0x%lx]"
#   define TMPL_BT_1    "(%s+0x%lx)"
#   define TMPL_BT_2    "[0x%lx]"
#   define TMPL_MEM     "%.16lx"
#   define TMPL_BLANK   "                 "
#else
#   define TMPL_REPEAT  "--- Address [0x%x] repeated %d times ---\n"
#   define TMPL_BT      "%s+0x%x [0x%x]"
#   define TMPL_BT_1    "(%s+0x%x)"
#   define TMPL_BT_2    "[0x%x]"
#   define TMPL_MEM     "%.8x"
#   define TMPL_BLANK   "         "
#endif

#ifdef __arm__
#   define EXT_REG_TMPL "LR: 0x"TMPL_MEM " FP: 0x"TMPL_MEM " SP: 0x"TMPL_MEM \
    "\n"
#else
#   define EXT_REG_TMPL "RA: 0x"TMPL_MEM " BP: 0x"TMPL_MEM " SP: 0x"TMPL_MEM \
    "\n"
#endif

struct exchnd_maps {
    unsigned long start;
    unsigned long end;
    unsigned int read : 1;
    unsigned int write : 1;
    unsigned int exec : 1;
    char name[512];
};

struct exchnd_unwind {
    unw_addr_space_t addr_space;
    void *upt_info;
    unw_cursor_t cursor;
    unw_accessors_t *accessors;
    /* Must be at the end: see libunwind_clean */
    struct exchnd_maps *maps;
    int num_maps;
};

/**
 * \func collect
 *
 * Prototype for collection function. Every collection function must
 * implement this interface.
 *
 * \param header Exception message header
 * \param data Data of the exception message
 * \param size Size of the data
 *
 * \return 0 if successful, error code otherwise
 */
typedef int (*collect)(ExchndInterface_t *,
                       struct exchnd_message_header *,
                       char *);

void exchnd_add_commandline(struct exchnd_message_header *header);
int libunwind_load(void);
void libunwind_unload(void);
void show_backtrace(void);

int get_func_from_stack(int, struct exchnd_unwind *);
int __log_backtrace(int, int, struct exchnd_unwind *);
int __collector_stack_dump(ExchndInterface_t *,
                           struct exchnd_message_header *,
                           char *);
int __collector_mem_dump(ExchndInterface_t *,
                         struct exchnd_message_header *,
                         unsigned long);

int pt_attach(pid_t, int);
int pt_group_attach(ExchndInterface_t *,
                    struct exchnd_message_header *,
                    int);
int pt_detach(int, int);

/* List of collector functions, collect data in user space */
extern collect collector_list[EHM_LAST_ELEMENT];

struct dump_mem {
    unw_word_t start;   /**< Start address */
    unw_word_t end;     /**< end address */
    unw_word_t add;     /**< Current address to dump */
    unw_word_t *line;   /**< A line of memory */
    unsigned int l;     /**< Number of lines to dump */
    unsigned int w;     /**< Words per line */
    char *phrase;       /**< Translation on line to printable characters */
    unw_word_t *out;    /**< Alternative output */
};

void exchnd_dump_mem(struct exchnd_unwind *, struct dump_mem *);

int exchnd_get_func_at_address(unw_word_t,
                               struct exchnd_unwind *,
                               char *,
                               int,
                               char *);

extern int (*unw_get_reg_p)(unw_cursor_t *, unw_regnum_t, unw_word_t *);

const char *libunwind_arch_lib(void);

int libunwind_load_arch(void *);
void libunwind_unload_arch(void);

unsigned int get_arch_first_reg(void);
unsigned int get_arch_last_reg(void);
unsigned int get_arch_pc_reg(void);
unsigned int get_arch_sp_reg(void);
unsigned int get_arch_lr_reg(void);
unsigned int get_arch_fp_reg(void);

void exchnd_extend_bt(struct exchnd_unwind *, unw_word_t *);

#define EXCHND_FB_NONE      0
#define EXCHND_FB_EXTENDED  1
#define EXCHND_FB_FRAME     2
#define EXCHND_FB_GUESS     3

#ifdef UNW_ARM_METHOD_ALL
#   define EXH_METHOD_FRAME    UNW_ARM_METHOD_FRAME
#   define EXH_METHOD_ALL      UNW_ARM_METHOD_ALL
#else /* !UNW_ARM_METHOD_ALL */
#   define EXH_METHOD_FRAME    0
#   define EXH_METHOD_ALL      0
#endif /* UNW_ARM_METHOD_ALL */

#endif